package navigation;

import cz.cuni.pogamut.Client.Agent;
import cz.cuni.pogamut.MessageObjects.Ammo;
import cz.cuni.pogamut.MessageObjects.Health;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.NeighNav;
import cz.cuni.pogamut.MessageObjects.Triple;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.logging.Level;

public class ThinkNavigation {
    protected Agent agent;
    protected double[][] distances;
    protected int[][] innerPoints;
    protected int id = 0;
    protected Map<NavPoint, Integer> navPoint2Id = new HashMap();
    protected ArrayList<NavPoint> navPoints;
    protected ArrayList<Long> unreachable = new ArrayList();
    protected IndexCube indexCube;
    protected LinkedList<Integer> ammos = new LinkedList();
    protected LinkedList<Integer> healths = new LinkedList();
    protected LinkedList<TIPair> takenAmmos = new LinkedList();
    protected LinkedList<TIPair> takenHealths = new LinkedList();
    protected Updator updator;

    protected class TIPair {
        public int pointId;
        public long timeTaken;
        
        public TIPair(int id) {
            pointId = id;
            timeTaken = System.currentTimeMillis();
        }
        
        public String toString() {
            return pointId + ": is there " + (System.currentTimeMillis() - timeTaken) + "ms";
        }
    }
    
    protected class Updator extends Thread {
        protected ThinkNavigation navigation;
        
        public Updator(ThinkNavigation nav) {
            super();
            navigation = nav;
        }
        
        @Override
        public void run() {
            while (true) {
                ArrayList<Ammo> seeAmmos = agent.getMemory().getSeeAmmos();
                ArrayList<Health> seeHealths = agent.getMemory().getSeeHealths();
                ArrayList<NavPoint> seeNavPoints = agent.getMemory().getSeeNavPoints();
                for (NavPoint p: seeNavPoints) {
                    if (p.item != null) {
                        // I wish the item.type worked :-(
                        if (p.item.getUnrealID().toLowerCase().contains("health")) {
                            boolean found = false;
                            for (Health h: seeHealths) {
                                if (h.navPoint.equals(p)) {
                                    found = true; 
                                }
                            }
                            if (!found) {
                                setHealthMissing(p);
                            }
                        } else if (p.item.getUnrealID().toLowerCase().contains("ammo")) {
                            boolean found = false;
                            for (Ammo a: seeAmmos) {
                                if (a.navPoint.equals(p)) {
                                   found = true; 
                                }
                            }
                            if (!found) {
                                setAmmoMissing(p);
                            }
                        }           
                    }
                }
                try {
                    Thread.sleep(200);
                } catch (InterruptedException ex) {
                    agent.getPlatformLog().log(Level.INFO, "Updator was interrupted.");
                    return;
                }
            }
        }
    }
      
    public ThinkNavigation(Agent agent) {
        this.agent = agent;
        agent.log.info("Initializing ThinkNavigation...");
        navPoints = agent.getMemory().getKnownNavPoints();
        agent.log.info("Filling the indexing cube...");
        // computing the size
        double indexCubeSize = 0;
        for (NavPoint p: navPoints) {
            indexCubeSize = absMax(indexCubeSize, p.location);
        }
        // but we'll put there some more navpoints, need space for them:
        for (Ammo a: agent.getMemory().getKnownAmmos()) {
            indexCubeSize = absMax(indexCubeSize, a.location);
        }
        for (Health h: agent.getMemory().getKnownHealths()) {
            indexCubeSize = absMax(indexCubeSize, h.location);
        }
        indexCube = new IndexCube(new Triple(0, 0, 0), indexCubeSize);
        for (NavPoint p: navPoints) {
            indexCube.insertNavPoint(p);
        }
        // index navPoints
        for (NavPoint p: navPoints) {
            navPoint2Id.put(p, id);
            unreachable.add(new Long(0));
            ++id;
        }
        // index healthpacks, ammos and so on
        // some items have not defined navpoint :-(
        for (Ammo a: agent.getMemory().getKnownAmmos()) {
            int npId;
            if (a.navPoint == null) {
                npId = createNavPoint(a.location);
            } else {
                npId = navPoint2Id.get(a.navPoint);
            }
            ammos.add(npId);
        }
        for (Health h: agent.getMemory().getKnownHealths()) {
            int npId;
            if (h.navPoint == null) {
                npId = createNavPoint(h.location);
            } else {
                npId = navPoint2Id.get(h.navPoint);
            }
            healths.add(npId);
        }
        agent.log.info("Indexing cube filled");
        agent.log.info("Starting computing Floyd-Warshall...");
        int n = navPoints.size();
        distances = new double[n][n];
        innerPoints = new int[n][n];
        int i = 0, j = 0;
        for (NavPoint p: navPoints) {
            for (j = 0; j < n; ++j) {
                distances[i][j] = Double.MAX_VALUE;
                innerPoints[i][j] = 0;
            }
            for(NeighNav nn: p.neighbours) {
                distances[i][navPoint2Id.get(nn.neighbour).intValue()] =
                        Triple.distanceInSpace(p.location, nn.neighbour.location);
            }
            distances[i][i] = 0;
            ++i;
        }
        agent.log.info("Array filled...");
        int k = 0;
        for (k = 0; k < n; ++k) {
            for (i = 0; i < n; ++i) {
                for (j = 0; j < n; ++j) {
                    if (distances[i][j] > distances[i][k] + distances[k][j]) {
                        distances[i][j] = distances[i][k] + distances[k][j];
                        innerPoints[i][j] = k;
                    }
                }
            }
        }
        agent.log.info("Floyd-Warshall finished.");
        updator = new Updator(this);
        updator.start();
        agent.log.info("ThinkNavigation initialized.");
    }

    public synchronized NavPoint getNearestAmmo(NavPoint navPoint, boolean includeSpecifiedPoint) {
        reactivate(27500, takenAmmos, ammos);
        return getNearestFrom(navPoint, ammos, includeSpecifiedPoint);
    }
    
    public synchronized NavPoint getNearestHealth(NavPoint navPoint, boolean includeSpecifiedPoint) {
        reactivate(27500, takenHealths, healths);
        return getNearestFrom(navPoint, healths, includeSpecifiedPoint);
    }
    
    public NavPoint getNearestNavPoint(Triple location) {
        for (double distance = 300; distance < 4*indexCube.getRadius(); distance += 200) {
            ArrayList<NavPoint> nearPoints = indexCube.getNavPointsNear(location, distance);
            if (!nearPoints.isEmpty()) {
                double minDistance = Double.MAX_VALUE;
                NavPoint nearest = null;
                for (NavPoint p: nearPoints) {
                    double d = Triple.distanceInSpace(p.location, location);
                    if (d < minDistance) {
                        nearest = p;
                        minDistance = d;
                    }
                }
                return nearest;
            }
        }
        return null;
    }

    public ArrayList<NavPoint> getPath(NavPoint p1, NavPoint p2) {
        if (p1 == null || p2 == null) {
            agent.getPlatformLog().log(Level.WARNING, "Navigation got invalid params (null):\n" + p1 + "\n"+ p2);
            return null;
        } else if (p1 == p2) {
            ArrayList nodes = new ArrayList();
            nodes.add(p1);
            return nodes;
        }
        int p1id = navPoint2Id.get(p1);
        int p2id = navPoint2Id.get(p2);
        ArrayList<NavPoint> nodes = new ArrayList();
        nodes.add(p1);
        fillPath(p1id, p2id, nodes);
        nodes.add(p2);
        return nodes;
    }
    
    public void killUpdator() {
        updator.interrupt();
    }
    
    public synchronized void setAmmoMissing(NavPoint p) {
        int pId = navPoint2Id.get(p);
        if (ammos.remove(new Integer(pId))) {
            takenAmmos.add(new TIPair(pId));
        }
    }
    
    public synchronized void setHealthMissing(NavPoint p) {
        int pId = navPoint2Id.get(p);
        if (healths.remove(new Integer(pId))) {
            takenHealths.add(new TIPair(pId));
        }
    }
    
    public synchronized void setNavPointUnreachable(NavPoint p) {
        unreachable.set(navPoint2Id.get(p).intValue(), System.currentTimeMillis());
    }
     
    private static double absMax(double currMax, Triple location) {
        if (currMax < Math.abs(location.x)) currMax = Math.abs(location.x);
        if (currMax < Math.abs(location.y)) currMax = Math.abs(location.y);
        if (currMax < Math.abs(location.z)) currMax = Math.abs(location.z);
        return currMax;
    }

    // this should be more complex and fill the navpoint at all
    private int createNavPoint(Triple location) {
        NavPoint p = new NavPoint();
        p.ID = 0;
        p.location = location;
        p.neighbours = new ArrayList();
        ArrayList<NavPoint> close = null;
        for (double dist = 200; dist < 2000; dist += 100) {
            close = indexCube.getNavPointsNear(p.location, dist);
            if (close.size() >= 2) break;
        }
        for (NavPoint n: close) {
            NeighNav nn = new NeighNav();
            nn.neighbour = n;
            p.neighbours.add(nn);
        }
        navPoints.add(p);
        unreachable.add(new Long(0));
        navPoint2Id.put(p, id);
        indexCube.insertNavPoint(p);
        return id++;
    }
    
    private NavPoint getNearestFrom(NavPoint navPoint, LinkedList<Integer> navPointIds, boolean includeSpecifiedPoint) {
        int navPointId = navPoint2Id.get(navPoint);
        double minDistance = Double.MAX_VALUE;
        int nearestId = -1;
        for (Integer i: navPointIds) {
            if (isUnreachable(navPointId)) continue;
            if (distances[navPointId][i.intValue()] < minDistance
                    && (includeSpecifiedPoint || i.intValue() != navPointId)) {
                nearestId = i.intValue();
                minDistance = distances[navPointId][nearestId];
            }
        }
        if (nearestId == -1) {
            return null;
        } else {
            return navPoints.get(nearestId);
        }
    }   

    private void fillPath(int p1, int p2, ArrayList<NavPoint> nodes) {
        int inner = innerPoints[p1][p2];
        if (inner != 0) {
            fillPath(p1, inner, nodes);
            nodes.add(navPoints.get(inner));
            fillPath(inner, p2, nodes);
        }
    }
    
    private boolean isUnreachable(int navPointId) {
        return unreachable.get(navPointId).longValue() + 30000 >
                System.currentTimeMillis();
    }
        
    /*
     * Deactivated items (e.g. I saw that someone took it) will reappear after
     * the activationTime - be optimistic and think as they will be there 
     */
    private void reactivate(long activationTime, LinkedList<TIPair> deactivated, LinkedList<Integer> active) {
        int reactivated = 0;
        for (TIPair tip: deactivated) {
            if (tip.timeTaken + activationTime > System.currentTimeMillis()) break;
            active.add(tip.pointId);
            reactivated++;
        }
        while (reactivated > 0) {
            deactivated.removeFirst();
            --reactivated;
        }        
    }
}